//
//  ItemGroups.swift
//  Do It
//
//  Created by Jim Dovey on 2/11/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI

enum TodoItemGroup {
    case today
    case scheduled
    case overdue
    case all
    
    var title: LocalizedStringKey {
        switch self {
        case .today: return "Today"
        case .scheduled: return "Scheduled"
        case .overdue: return "Overdue"
        case .all: return "All"
        }
    }
    
    var color: Color {
        switch self {
        case .today: return .blue
        case .scheduled: return .orange
        case .overdue: return .red
        case .all: return Color(.darkGray)
        }
    }
    
    private var image: some View {
        switch self {
        case .today:
            return iconImage("calendar")
                .offset(x: 0, y: 0)
        case .scheduled:
            return iconImage("clock.fill")
                .offset(x: 0, y: 0)
        case .overdue:
            return iconImage("exclamationmark.triangle.fill")
                .offset(x: 0, y: -2)
        case .all:
            return iconImage("tray.fill")
                .offset(x: 0, y: -1)
        }
    }
    
    var icon: some View {
        self.image
            .font(.system(.body, design: .rounded))
            .foregroundColor(.white)
            .padding(10)
            .background(color.clipShape(Circle()))
    }
    
    func items(from data: DataCenter) -> [TodoItem] {
        switch self {
        case .today:
            return todayItems(data.todoItems)
        case .scheduled:
            return scheduledItems(data.todoItems)
        case .overdue:
            return overdueItems(data.todoItems)
        case .all:
            return data.todoItems
        }
    }
    
    private func todayItems(_ items: [TodoItem]) -> [TodoItem] {
        let todayComponents = Calendar.current
            .dateComponents([.day, .month, .year], from: Date())
        let today = Calendar.current.date(from: todayComponents)!
        let interval = DateInterval(start: today, duration: 60 * 60 * 24)
        
        return items.filter {
            $0.date != nil && interval.contains($0.date!)
        }
    }
    
    private func scheduledItems(_ items: [TodoItem]) -> [TodoItem] {
        items.filter { $0.date != nil && !$0.complete }
    }
    
    private func overdueItems(_ items: [TodoItem]) -> [TodoItem] {
        items.filter {
            $0.complete == false && $0.date != nil && $0.date! < Date()
        }
    }
}
